<?php
// view_vps.php
require_once 'config.php';
redirectIfNotLoggedIn();
redirectIfNotAdmin(); // Only admins can access this page

require_once 'contabo_api.php';

$message = '';
$error = '';
$instances = [];
$users = [];

try {
    $contabo = new ContaboAPI();
    
    // Handle instance actions
    if (isset($_GET['action']) && isset($_GET['instance_id'])) {
        $action = $_GET['action'];
        $instanceId = $_GET['instance_id'];
        
        switch($action) {
            case 'start':
                $contabo->startInstance($instanceId);
                $message = "Instance started successfully!";
                break;
            case 'stop':
                $contabo->stopInstance($instanceId);
                $message = "Instance stopped successfully!";
                break;
            case 'restart':
                $contabo->restartInstance($instanceId);
                $message = "Instance restarted successfully!";
                break;
            case 'assign':
                if (isset($_GET['user_id'])) {
                    $userId = $_GET['user_id'];
                    $db = getDB();
                    
                    // Check if assignment already exists
                    $checkStmt = $db->prepare("SELECT id FROM user_vps_assignments WHERE user_id = ? AND instance_id = ?");
                    $checkStmt->execute([$userId, $instanceId]);
                    
                    if (!$checkStmt->fetch()) {
                        $stmt = $db->prepare("INSERT INTO user_vps_assignments (user_id, instance_id) VALUES (?, ?)");
                        $stmt->execute([$userId, $instanceId]);
                        
                        // Also update the vps_instances table
                        $updateStmt = $db->prepare("UPDATE vps_instances SET assigned_to = ? WHERE instance_id = ?");
                        $updateStmt->execute([$userId, $instanceId]);
                        
                        $message = "VPS assigned to user successfully!";
                    } else {
                        $message = "VPS is already assigned to this user!";
                    }
                }
                break;
            case 'unassign':
                $db = getDB();
                $stmt = $db->prepare("DELETE FROM user_vps_assignments WHERE instance_id = ?");
                $stmt->execute([$instanceId]);
                
                // Also update the vps_instances table
                $updateStmt = $db->prepare("UPDATE vps_instances SET assigned_to = NULL WHERE instance_id = ?");
                $updateStmt->execute([$instanceId]);
                
                $message = "VPS unassigned successfully!";
                break;
        }
    }
    
    // Check if we need to sync
    if (isset($_GET['sync']) && $_GET['sync'] === 'true') {
        $count = $contabo->syncInstancesToDB();
        $message = "Successfully synced $count instances from Contabo API!";
    }
    
    // Load instances from database
    $db = getDB();
    $stmt = $db->query("
        SELECT vi.*, u.username as assigned_username 
        FROM vps_instances vi 
        LEFT JOIN users u ON vi.assigned_to = u.id 
        ORDER BY vi.created_date DESC
    ");
    $instances = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
    // Load users for assignment
    $userStmt = $db->query("SELECT id, username FROM users WHERE role = 'user' ORDER BY username");
    $users = $userStmt->fetchAll(PDO::FETCH_ASSOC);
    
} catch(Exception $e) {
    $error = $e->getMessage();
}
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Manage VPS Instances - Admin</title>
    <style>
        body { font-family: Arial, sans-serif; margin: 20px; }
        .message { padding: 10px; margin: 10px 0; border-radius: 4px; }
        .success { background: #d4edda; color: #155724; border: 1px solid #c3e6cb; }
        .error { background: #f8d7da; color: #721c24; border: 1px solid #f5c6cb; }
        table { width: 100%; border-collapse: collapse; margin-top: 20px; }
        th, td { padding: 12px; text-align: left; border-bottom: 1px solid #ddd; }
        th { background-color: #f8f9fa; font-weight: bold; }
        .status-running { color: #28a745; font-weight: bold; }
        .status-stopped { color: #dc3545; font-weight: bold; }
        .status-provisioning { color: #ffc107; font-weight: bold; }
        .btn { display: inline-block; padding: 6px 12px; margin: 2px; border: none; border-radius: 4px; cursor: pointer; text-decoration: none; font-size: 12px; }
        .btn-success { background: #28a745; color: white; }
        .btn-warning { background: #ffc107; color: black; }
        .btn-danger { background: #dc3545; color: white; }
        .btn-info { background: #17a2b8; color: white; }
        .btn-primary { background: #007bff; color: white; }
        .btn-secondary { background: #6c757d; color: white; }
        .header { display: flex; justify-content: space-between; align-items: center; margin-bottom: 20px; flex-wrap: wrap; }
        .actions { display: flex; gap: 5px; flex-wrap: wrap; }
        .user-assignment { margin: 10px 0; }
        .assignment-form { display: inline-block; margin-left: 10px; }
    </style>
</head>
<body>
    <div class="header">
        <h1>Manage VPS Instances (Admin)</h1>
        <div>
            <span>Welcome, <?php echo htmlspecialchars($_SESSION['username']); ?> (Admin)</span>
            <a href="user_vps.php" class="btn btn-info">User View</a>
            <a href="settings.php" class="btn btn-primary">Settings</a>
            <a href="view_vps.php?sync=true" class="btn btn-success">Sync with Contabo</a>
            <a href="logout.php" class="btn btn-secondary">Logout</a>
        </div>
    </div>
    
    <?php if ($message): ?>
        <div class="message success"><?php echo htmlspecialchars($message); ?></div>
    <?php endif; ?>
    
    <?php if ($error): ?>
        <div class="message error"><?php echo htmlspecialchars($error); ?></div>
    <?php endif; ?>
    
    <?php if (empty($instances)): ?>
        <p>No VPS instances found. <a href="view_vps.php?sync=true">Sync with Contabo</a> to load your instances.</p>
    <?php else: ?>
        <table>
            <thead>
                <tr>
                    <th>Name</th>
                    <th>Instance ID</th>
                    <th>Status</th>
                    <th>IP Address</th>
                    <th>Region</th>
                    <th>Assigned To</th>
                    <th>Created Date</th>
                    <th>Actions</th>
                </tr>
            </thead>
            <tbody>
                <?php foreach ($instances as $instance): ?>
                <tr>
                    <td><?php echo htmlspecialchars($instance['name']); ?></td>
                    <td><?php echo htmlspecialchars($instance['instance_id']); ?></td>
                    <td class="status-<?php echo strtolower($instance['status']); ?>">
                        <?php echo htmlspecialchars($instance['status']); ?>
                    </td>
                    <td><?php echo htmlspecialchars($instance['ip_address']); ?></td>
                    <td><?php echo htmlspecialchars($instance['region']); ?></td>
                    <td>
                        <?php if ($instance['assigned_username']): ?>
                            <?php echo htmlspecialchars($instance['assigned_username']); ?>
                            <a href="view_vps.php?action=unassign&instance_id=<?php echo $instance['instance_id']; ?>" 
                               class="btn btn-danger btn-sm" 
                               onclick="return confirm('Unassign this VPS from user?')">Unassign</a>
                        <?php else: ?>
                            <div class="user-assignment">
                                <span>Not assigned</span>
                                <div class="assignment-form">
                                    <form method="GET" style="display: inline;">
                                        <input type="hidden" name="action" value="assign">
                                        <input type="hidden" name="instance_id" value="<?php echo $instance['instance_id']; ?>">
                                        <select name="user_id" onchange="this.form.submit()" style="padding: 4px;">
                                            <option value="">Assign to user...</option>
                                            <?php foreach ($users as $user): ?>
                                                <option value="<?php echo $user['id']; ?>"><?php echo htmlspecialchars($user['username']); ?></option>
                                            <?php endforeach; ?>
                                        </select>
                                    </form>
                                </div>
                            </div>
                        <?php endif; ?>
                    </td>
                    <td><?php echo date('M j, Y H:i', strtotime($instance['created_date'])); ?></td>
                    <td class="actions">
                        <a href="view_vps.php?action=start&instance_id=<?php echo $instance['instance_id']; ?>" class="btn btn-success" onclick="return confirm('Start this instance?')">Start</a>
                        <a href="view_vps.php?action=stop&instance_id=<?php echo $instance['instance_id']; ?>" class="btn btn-danger" onclick="return confirm('Stop this instance?')">Stop</a>
                        <a href="view_vps.php?action=restart&instance_id=<?php echo $instance['instance_id']; ?>" class="btn btn-warning" onclick="return confirm('Restart this instance?')">Restart</a>
                        <a href="instance_details.php?instance_id=<?php echo $instance['instance_id']; ?>" class="btn btn-info">Details</a>
                    </td>
                </tr>
                <?php endforeach; ?>
            </tbody>
        </table>
        <p><small>Last sync: <?php echo date('M j, Y H:i:s'); ?></small></p>
    <?php endif; ?>
</body>
</html>