<?php
// user_vps.php
require_once 'config.php';
redirectIfNotLoggedIn();

// Different redirect based on role
if (isCustomer()) {
    header("Location: customer_vps.php");
    exit;
}

$message = '';
$error = '';
$user_instances = [];
$customers = [];

try {
    $db = getDB();
    
    // Get VPS instances assigned to the current user
    $stmt = $db->prepare("
        SELECT vi.*, uva.customer_id, c.username as customer_username
        FROM vps_instances vi 
        INNER JOIN user_vps_assignments uva ON vi.instance_id = uva.instance_id 
        LEFT JOIN customers c ON uva.customer_id = c.id
        WHERE uva.user_id = ? 
        ORDER BY vi.created_date DESC
    ");
    $stmt->execute([$_SESSION['user_id']]);
    $user_instances = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
    // Get customers for this store
    $customerStmt = $db->prepare("SELECT id, username, email FROM customers WHERE store_user_id = ? AND status = 'active' ORDER BY username");
    $customerStmt->execute([$_SESSION['user_id']]);
    $customers = $customerStmt->fetchAll(PDO::FETCH_ASSOC);
    
    // Handle instance actions
    if (isset($_GET['action']) && isset($_GET['instance_id'])) {
        $action = $_GET['action'];
        $instanceId = $_GET['instance_id'];
        
        // Verify user has access to this instance
        $checkStmt = $db->prepare("
            SELECT 1 FROM user_vps_assignments 
            WHERE user_id = ? AND instance_id = ?
        ");
        $checkStmt->execute([$_SESSION['user_id'], $instanceId]);
        
        if ($checkStmt->fetch()) {
            $contabo = new ContaboAPI();
            
            switch($action) {
                case 'start':
                    $contabo->startInstance($instanceId);
                    $contabo->logAction($instanceId, 'start');
                    $message = "Instance started successfully!";
                    break;
                case 'stop':
                    $contabo->stopInstance($instanceId);
                    $contabo->logAction($instanceId, 'stop');
                    $message = "Instance stopped successfully!";
                    break;
                case 'restart':
                    $contabo->restartInstance($instanceId);
                    $contabo->logAction($instanceId, 'restart');
                    $message = "Instance restarted successfully!";
                    break;
                case 'assign_customer':
                    if (isset($_GET['customer_id'])) {
                        $customerId = $_GET['customer_id'];
                        $updateStmt = $db->prepare("UPDATE user_vps_assignments SET customer_id = ? WHERE instance_id = ? AND user_id = ?");
                        $updateStmt->execute([$customerId, $instanceId, $_SESSION['user_id']]);
                        $message = "VPS assigned to customer successfully!";
                    }
                    break;
                case 'unassign_customer':
                    $updateStmt = $db->prepare("UPDATE user_vps_assignments SET customer_id = NULL WHERE instance_id = ? AND user_id = ?");
                    $updateStmt->execute([$instanceId, $_SESSION['user_id']]);
                    $message = "VPS unassigned from customer successfully!";
                    break;
            }
        } else {
            $error = "You don't have permission to perform this action on this instance.";
        }
    }
    
} catch(Exception $e) {
    $error = $e->getMessage();
}
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>My VPS - <?php echo htmlspecialchars($_SESSION['username']); ?></title>
    <style>
        body { font-family: Arial, sans-serif; margin: 20px; }
        .message { padding: 10px; margin: 10px 0; border-radius: 4px; }
        .success { background: #d4edda; color: #155724; border: 1px solid #c3e6cb; }
        .error { background: #f8d7da; color: #721c24; border: 1px solid #f5c6cb; }
        table { width: 100%; border-collapse: collapse; margin-top: 20px; }
        th, td { padding: 12px; text-align: left; border-bottom: 1px solid #ddd; }
        th { background-color: #f8f9fa; font-weight: bold; }
        .status-running { color: #28a745; font-weight: bold; }
        .status-stopped { color: #dc3545; font-weight: bold; }
        .status-provisioning { color: #ffc107; font-weight: bold; }
        .btn { display: inline-block; padding: 6px 12px; margin: 2px; border: none; border-radius: 4px; cursor: pointer; text-decoration: none; font-size: 12px; }
        .btn-success { background: #28a745; color: white; }
        .btn-warning { background: #ffc107; color: black; }
        .btn-danger { background: #dc3545; color: white; }
        .btn-info { background: #17a2b8; color: white; }
        .btn-primary { background: #007bff; color: white; }
        .btn-secondary { background: #6c757d; color: white; }
        .header { display: flex; justify-content: space-between; align-items: center; margin-bottom: 20px; flex-wrap: wrap; }
        .actions { display: flex; gap: 5px; flex-wrap: wrap; }
        .customer-assignment { margin: 5px 0; }
        .nav-links { margin-bottom: 20px; }
    </style>
</head>
<body>
    <div class="nav-links">
        <a href="user_vps.php">My VPS</a> | 
        <a href="user_customers.php">My Customers</a> | 
        <a href="user_store_settings.php">Store Settings</a> | 
        <a href="logout.php">Logout</a>
    </div>

    <div class="header">
        <h1>My VPS Instances</h1>
        <div>
            <span>Welcome, <?php echo htmlspecialchars($_SESSION['username']); ?> (Store Owner)</span>
            <?php if (isAdmin()): ?>
                <a href="view_vps.php" class="btn btn-info">Admin View</a>
            <?php endif; ?>
        </div>
    </div>
    
    <?php if ($message): ?>
        <div class="message success"><?php echo htmlspecialchars($message); ?></div>
    <?php endif; ?>
    
    <?php if ($error): ?>
        <div class="message error"><?php echo htmlspecialchars($error); ?></div>
    <?php endif; ?>
    
    <?php if (empty($user_instances)): ?>
        <p>No VPS instances assigned to your store yet.</p>
        <p>Please contact administrator to assign VPS instances to your store.</p>
    <?php else: ?>
        <table>
            <thead>
                <tr>
                    <th>Name</th>
                    <th>Instance ID</th>
                    <th>Status</th>
                    <th>IP Address</th>
                    <th>Assigned To Customer</th>
                    <th>Created Date</th>
                    <th>Actions</th>
                </tr>
            </thead>
            <tbody>
                <?php foreach ($user_instances as $instance): ?>
                <tr>
                    <td><?php echo htmlspecialchars($instance['name']); ?></td>
                    <td><?php echo htmlspecialchars($instance['instance_id']); ?></td>
                    <td class="status-<?php echo strtolower($instance['status']); ?>">
                        <?php echo htmlspecialchars($instance['status']); ?>
                    </td>
                    <td><?php echo htmlspecialchars($instance['ip_address']); ?></td>
                    <td>
                        <?php if ($instance['customer_username']): ?>
                            <?php echo htmlspecialchars($instance['customer_username']); ?>
                            <a href="user_vps.php?action=unassign_customer&instance_id=<?php echo $instance['instance_id']; ?>" 
                               class="btn btn-danger btn-sm" 
                               onclick="return confirm('Unassign this VPS from customer?')">Unassign</a>
                        <?php else: ?>
                            <div class="customer-assignment">
                                <span>Not assigned</span>
                                <?php if (!empty($customers)): ?>
                                    <div>
                                        <form method="GET" style="display: inline;">
                                            <input type="hidden" name="action" value="assign_customer">
                                            <input type="hidden" name="instance_id" value="<?php echo $instance['instance_id']; ?>">
                                            <select name="customer_id" onchange="this.form.submit()" style="padding: 4px;">
                                                <option value="">Assign to customer...</option>
                                                <?php foreach ($customers as $customer): ?>
                                                    <option value="<?php echo $customer['id']; ?>"><?php echo htmlspecialchars($customer['username']); ?></option>
                                                <?php endforeach; ?>
                                            </select>
                                        </form>
                                    </div>
                                <?php else: ?>
                                    <div><small>No customers available</small></div>
                                <?php endif; ?>
                            </div>
                        <?php endif; ?>
                    </td>
                    <td><?php echo date('M j, Y H:i', strtotime($instance['created_date'])); ?></td>
                    <td class="actions">
                        <a href="user_vps.php?action=start&instance_id=<?php echo $instance['instance_id']; ?>" class="btn btn-success" onclick="return confirm('Start this instance?')">Start</a>
                        <a href="user_vps.php?action=stop&instance_id=<?php echo $instance['instance_id']; ?>" class="btn btn-danger" onclick="return confirm('Stop this instance?')">Stop</a>
                        <a href="user_vps.php?action=restart&instance_id=<?php echo $instance['instance_id']; ?>" class="btn btn-warning" onclick="return confirm('Restart this instance?')">Restart</a>
                        <a href="instance_details.php?instance_id=<?php echo $instance['instance_id']; ?>" class="btn btn-info">Manage</a>
                    </td>
                </tr>
                <?php endforeach; ?>
            </tbody>
        </table>
    <?php endif; ?>
</body>
</html>