<?php
// user_store_settings.php
require_once 'config.php';
redirectIfNotLoggedIn();
redirectIfNotUser();

$message = '';
$error = '';
$store_settings = [];

try {
    $db = getDB();
    
    // Get current store settings
    $stmt = $db->prepare("SELECT * FROM user_store_settings WHERE user_id = ?");
    $stmt->execute([$_SESSION['user_id']]);
    $store_settings = $stmt->fetch(PDO::FETCH_ASSOC);
    
    if (!$store_settings) {
        throw new Exception("Store settings not found. Please contact administrator.");
    }
    
    // Handle form submission
    if ($_SERVER['REQUEST_METHOD'] === 'POST') {
        $store_name = trim($_POST['store_name'] ?? '');
        $store_domain = trim($_POST['store_domain'] ?? '');
        $store_description = trim($_POST['store_description'] ?? '');
        $store_theme = $_POST['store_theme'] ?? 'default';
        
        // Validation
        if (empty($store_name)) {
            $error = "Store name is required!";
        } elseif (empty($store_domain)) {
            $error = "Store domain is required!";
        } elseif (!preg_match('/^[a-zA-Z0-9.-]+\.[a-zA-Z]{2,}$/', $store_domain)) {
            $error = "Please enter a valid domain name (e.g., mystore.example.com)";
        } else {
            // Check if domain is already taken by another store
            if ($store_domain !== $store_settings['store_domain']) {
                $checkStmt = $db->prepare("SELECT id FROM user_store_settings WHERE store_domain = ? AND user_id != ?");
                $checkStmt->execute([$store_domain, $_SESSION['user_id']]);
                
                if ($checkStmt->fetch()) {
                    $error = "This domain is already taken by another store!";
                }
            }
            
            if (!$error) {
                $updateStmt = $db->prepare("
                    UPDATE user_store_settings 
                    SET store_name = ?, store_domain = ?, store_description = ?, store_theme = ?, updated_at = NOW() 
                    WHERE user_id = ?
                ");
                $updateStmt->execute([$store_name, $store_domain, $store_description, $store_theme, $_SESSION['user_id']]);
                
                $message = "Store settings updated successfully!";
                
                // Refresh store settings
                $stmt->execute([$_SESSION['user_id']]);
                $store_settings = $stmt->fetch(PDO::FETCH_ASSOC);
                
                // Update session if domain changed
                if (isset($_SESSION['store_domain']) && $store_domain !== $_SESSION['store_domain']) {
                    $_SESSION['store_domain'] = $store_domain;
                }
            }
        }
    }
    
} catch(Exception $e) {
    $error = $e->getMessage();
}
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Store Settings - VPS Manager</title>
    <style>
        body { font-family: Arial, sans-serif; margin: 20px; }
        .message { padding: 10px; margin: 10px 0; border-radius: 4px; }
        .success { background: #d4edda; color: #155724; border: 1px solid #c3e6cb; }
        .error { background: #f8d7da; color: #721c24; border: 1px solid #f5c6cb; }
        .form-group { margin-bottom: 15px; }
        label { display: block; margin-bottom: 5px; font-weight: bold; }
        input[type="text"], textarea, select { 
            width: 100%; max-width: 500px; padding: 8px; border: 1px solid #ddd; border-radius: 4px; 
            box-sizing: border-box;
        }
        button { background: #007bff; color: white; padding: 10px 20px; border: none; border-radius: 4px; cursor: pointer; margin-right: 10px; }
        .nav-links { margin-bottom: 20px; }
        .store-info { background: #f8f9fa; padding: 15px; border-radius: 5px; margin-bottom: 20px; }
        .form-section { background: white; padding: 20px; border-radius: 5px; border: 1px solid #dee2e6; margin-bottom: 20px; }
        .form-section h3 { margin-top: 0; color: #495057; border-bottom: 1px solid #dee2e6; padding-bottom: 10px; }
        .help-text { color: #6c757d; font-size: 0.875em; margin-top: 5px; }
        .danger-zone { background: #f8d7da; border: 1px solid #f5c6cb; padding: 20px; border-radius: 5px; margin-top: 30px; }
        .danger-zone h3 { color: #721c24; margin-top: 0; }
        .btn-danger { background: #dc3545; color: white; }
        .current-url { background: #e9ecef; padding: 10px; border-radius: 4px; margin: 10px 0; font-family: monospace; }
    </style>
</head>
<body>
    <div class="nav-links">
        <a href="user_vps.php">My VPS</a> | 
        <a href="user_customers.php">My Customers</a> | 
        <a href="user_store_settings.php">Store Settings</a> | 
        <a href="logout.php">Logout</a>
    </div>

    <h1>Store Settings</h1>
    
    <?php if ($message): ?>
        <div class="message success"><?php echo htmlspecialchars($message); ?></div>
    <?php endif; ?>
    
    <?php if ($error): ?>
        <div class="message error"><?php echo htmlspecialchars($error); ?></div>
    <?php endif; ?>
    
    <div class="store-info">
        <h3>Store Information</h3>
        <p><strong>Current Store Domain:</strong> <?php echo htmlspecialchars($store_settings['store_domain']); ?></p>
        <p><strong>Current Store URL:</strong> 
            <div class="current-url">http://<?php echo htmlspecialchars($store_settings['store_domain']); ?>/store_dashboard.php</div>
        </p>
        <p><strong>Created:</strong> <?php echo date('M j, Y H:i', strtotime($store_settings['created_at'])); ?></p>
        <p><strong>Last Updated:</strong> <?php echo date('M j, Y H:i', strtotime($store_settings['updated_at'])); ?></p>
    </div>

    <div class="form-section">
        <h3>Basic Store Settings</h3>
        <form method="POST">
            <div class="form-group">
                <label for="store_name">Store Name:</label>
                <input type="text" id="store_name" name="store_name" 
                       value="<?php echo htmlspecialchars($store_settings['store_name']); ?>" required>
                <div class="help-text">The display name of your store that customers will see</div>
            </div>
            
            <div class="form-group">
                <label for="store_domain">Store Domain:</label>
                <input type="text" id="store_domain" name="store_domain" 
                       value="<?php echo htmlspecialchars($store_settings['store_domain']); ?>" required>
                <div class="help-text">
                    Your store domain (e.g., mystore.example.com). 
                    Customers will access your store at: http://<?php echo htmlspecialchars($store_settings['store_domain']); ?>/store_dashboard.php
                </div>
            </div>
            
            <div class="form-group">
                <label for="store_description">Store Description:</label>
                <textarea id="store_description" name="store_description" rows="4" 
                          placeholder="Describe your store and services..."><?php echo htmlspecialchars($store_settings['store_description'] ?? ''); ?></textarea>
                <div class="help-text">A brief description of your store that will be shown to customers</div>
            </div>
            
            <div class="form-group">
                <label for="store_theme">Store Theme:</label>
                <select id="store_theme" name="store_theme">
                    <option value="default" <?php echo ($store_settings['store_theme'] ?? 'default') === 'default' ? 'selected' : ''; ?>>Default Theme</option>
                    <option value="dark" <?php echo ($store_settings['store_theme'] ?? 'default') === 'dark' ? 'selected' : ''; ?>>Dark Theme</option>
                    <option value="blue" <?php echo ($store_settings['store_theme'] ?? 'default') === 'blue' ? 'selected' : ''; ?>>Blue Theme</option>
                    <option value="green" <?php echo ($store_settings['store_theme'] ?? 'default') === 'green' ? 'selected' : ''; ?>>Green Theme</option>
                    <option value="purple" <?php echo ($store_settings['store_theme'] ?? 'default') === 'purple' ? 'selected' : ''; ?>>Purple Theme</option>
                    <option value="modern" <?php echo ($store_settings['store_theme'] ?? 'default') === 'modern' ? 'selected' : ''; ?>>Modern Theme</option>
                </select>
                <div class="help-text">Choose the color scheme for your store dashboard</div>
            </div>
            
            <button type="submit">Update Store Settings</button>
            <button type="button" onclick="window.location.reload()">Cancel</button>
        </form>
    </div>

    <div class="form-section">
        <h3>Store Integration</h3>
        <div class="form-group">
            <label>Store Dashboard URL:</label>
            <div class="current-url">http://<?php echo htmlspecialchars($store_settings['store_domain']); ?>/store_dashboard.php</div>
            <div class="help-text">Share this URL with your customers for registration and login</div>
        </div>
        
        <div class="form-group">
            <label>Embed Code (For Website):</label>
            <textarea readonly rows="3" style="width: 100%; font-family: monospace; font-size: 12px; background: #f8f9fa;">
<a href="http://<?php echo htmlspecialchars($store_settings['store_domain']); ?>/store_dashboard.php" target="_blank">
    VPS Hosting Services - <?php echo htmlspecialchars($store_settings['store_name']); ?>
</a>
            </textarea>
            <div class="help-text">Copy and paste this code on your website to link to your store</div>
        </div>
        
        <div class="form-group">
            <label>Direct Login URL for Customers:</label>
            <div class="current-url">http://<?php echo htmlspecialchars($store_settings['store_domain']); ?>/store_dashboard.php</div>
            <div class="help-text">Customers can use this URL to directly access your store</div>
        </div>
    </div>

    <div class="danger-zone">
        <h3>⚠️ Danger Zone</h3>
        <p><strong>Reset Store Data</strong></p>
        <p>This will remove all your customers and their assignments, but keep your VPS instances.</p>
        <button type="button" class="btn-danger" onclick="confirmReset()">Reset Store Data</button>
        
        <p style="margin-top: 20px;"><strong>Delete Store</strong></p>
        <p>This will permanently delete your store and all associated data. This action cannot be undone.</p>
        <button type="button" class="btn-danger" onclick="confirmDelete()">Delete Store</button>
    </div>

    <script>
        function confirmReset() {
            if (confirm('⚠️ WARNING: This will remove ALL your customers and their VPS assignments. Your VPS instances will remain assigned to you. This action cannot be undone.\n\nAre you sure you want to reset your store data?')) {
                // Implement store reset functionality
                alert('Store reset functionality would be implemented here. This is a placeholder.');
            }
        }
        
        function confirmDelete() {
            if (confirm('🚨 DANGER: This will PERMANENTLY DELETE your store, all customers, and their data. This action cannot be undone.\n\nAre you absolutely sure you want to delete your store?')) {
                // Implement store deletion functionality
                alert('Store deletion functionality would be implemented here. This is a placeholder.');
            }
        }
        
        // Domain validation
        document.getElementById('store_domain').addEventListener('blur', function() {
            const domain = this.value;
            const domainPattern = /^[a-zA-Z0-9.-]+\.[a-zA-Z]{2,}$/;
            
            if (domain && !domainPattern.test(domain)) {
                alert('Please enter a valid domain name (e.g., mystore.example.com)');
                this.focus();
            }
        });
        
        // Show preview of new URL when domain changes
        document.getElementById('store_domain').addEventListener('input', function() {
            const newDomain = this.value;
            const previewElement = document.getElementById('url-preview');
            
            if (!previewElement) {
                const helpText = this.parentNode.querySelector('.help-text');
                const preview = document.createElement('div');
                preview.id = 'url-preview';
                preview.className = 'current-url';
                preview.style.marginTop = '10px';
                preview.innerHTML = 'New URL: http://' + newDomain + '/store_dashboard.php';
                helpText.appendChild(preview);
            } else {
                previewElement.innerHTML = 'New URL: http://' + newDomain + '/store_dashboard.php';
            }
        });
    </script>
</body>
</html>