<?php
// user_management.php
require_once 'config.php';
redirectIfNotLoggedIn();
redirectIfNotAdmin();

$message = '';
$error = '';
$users = [];

try {
    $db = getDB();
    
    // Get all users with their store information
    $stmt = $db->prepare("
        SELECT u.*, uss.store_name, uss.store_domain,
               (SELECT COUNT(*) FROM customers c WHERE c.store_user_id = u.id) as customer_count,
               (SELECT COUNT(*) FROM user_vps_assignments uva WHERE uva.user_id = u.id) as vps_count
        FROM users u 
        LEFT JOIN user_store_settings uss ON u.id = uss.user_id 
        ORDER BY u.created_at DESC
    ");
    $stmt->execute();
    $users = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
    // Handle user actions
    if (isset($_GET['action']) && isset($_GET['user_id'])) {
        $action = $_GET['action'];
        $userId = $_GET['user_id'];
        
        switch($action) {
            case 'delete':
                // Only allow deleting non-admin users
                $checkStmt = $db->prepare("SELECT role FROM users WHERE id = ?");
                $checkStmt->execute([$userId]);
                $user = $checkStmt->fetch();
                
                if ($user && $user['role'] !== 'admin') {
                    $deleteStmt = $db->prepare("DELETE FROM users WHERE id = ?");
                    $deleteStmt->execute([$userId]);
                    $message = "User deleted successfully!";
                } else {
                    $error = "Cannot delete admin users!";
                }
                break;
        }
        
        // Refresh users list
        $stmt->execute();
        $users = $stmt->fetchAll(PDO::FETCH_ASSOC);
    }
    
} catch(Exception $e) {
    $error = $e->getMessage();
}
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>User Management - VPS Manager</title>
    <style>
        body { font-family: Arial, sans-serif; margin: 20px; }
        .message { padding: 10px; margin: 10px 0; border-radius: 4px; }
        .success { background: #d4edda; color: #155724; border: 1px solid #c3e6cb; }
        .error { background: #f8d7da; color: #721c24; border: 1px solid #f5c6cb; }
        table { width: 100%; border-collapse: collapse; margin-top: 20px; }
        th, td { padding: 12px; text-align: left; border-bottom: 1px solid #ddd; }
        th { background-color: #f8f9fa; font-weight: bold; }
        .btn { display: inline-block; padding: 6px 12px; margin: 2px; border: none; border-radius: 4px; cursor: pointer; text-decoration: none; font-size: 12px; }
        .btn-primary { background: #007bff; color: white; }
        .btn-danger { background: #dc3545; color: white; }
        .nav-links { margin-bottom: 20px; }
        .header { display: flex; justify-content: space-between; align-items: center; margin-bottom: 20px; }
        .role-admin { color: #dc3545; font-weight: bold; }
        .role-user { color: #007bff; font-weight: bold; }
        .role-customer { color: #28a745; font-weight: bold; }
    </style>
</head>
<body>
    <div class="nav-links">
        <a href="view_vps.php">VPS List</a> | 
        <a href="user_management.php">User Management</a> | 
        <a href="register_store_user.php">Register Store User</a> | 
        <a href="logout.php">Logout</a>
    </div>

    <div class="header">
        <h1>User Management</h1>
        <div>
            <a href="register_store_user.php" class="btn btn-primary">Register New Store User</a>
        </div>
    </div>
    
    <?php if ($message): ?>
        <div class="message success"><?php echo htmlspecialchars($message); ?></div>
    <?php endif; ?>
    
    <?php if ($error): ?>
        <div class="message error"><?php echo htmlspecialchars($error); ?></div>
    <?php endif; ?>
    
    <table>
        <thead>
            <tr>
                <th>Username</th>
                <th>Email</th>
                <th>Role</th>
                <th>Store Name</th>
                <th>Store Domain</th>
                <th>Customers</th>
                <th>VPS Count</th>
                <th>Registered</th>
                <th>Actions</th>
            </tr>
        </thead>
        <tbody>
            <?php foreach ($users as $user): ?>
            <tr>
                <td><?php echo htmlspecialchars($user['username']); ?></td>
                <td><?php echo htmlspecialchars($user['email']); ?></td>
                <td class="role-<?php echo $user['role']; ?>">
                    <?php echo htmlspecialchars(ucfirst($user['role'])); ?>
                </td>
                <td>
                    <?php echo $user['store_name'] ? htmlspecialchars($user['store_name']) : 'N/A'; ?>
                </td>
                <td>
                    <?php if ($user['store_domain']): ?>
                        <a href="http://<?php echo htmlspecialchars($user['store_domain']); ?>" target="_blank">
                            <?php echo htmlspecialchars($user['store_domain']); ?>
                        </a>
                    <?php else: ?>
                        N/A
                    <?php endif; ?>
                </td>
                <td><?php echo $user['customer_count'] ?? 0; ?></td>
                <td><?php echo $user['vps_count'] ?? 0; ?></td>
                <td><?php echo date('M j, Y H:i', strtotime($user['created_at'])); ?></td>
                <td>
                    <?php if ($user['role'] !== 'admin'): ?>
                        <a href="user_management.php?action=delete&user_id=<?php echo $user['id']; ?>" 
                           class="btn btn-danger" 
                           onclick="return confirm('Delete this user? This will also delete their store and customers.')">Delete</a>
                    <?php else: ?>
                        <span style="color: #6c757d;">System</span>
                    <?php endif; ?>
                </td>
            </tr>
            <?php endforeach; ?>
        </tbody>
    </table>
</body>
</html>