<?php

// instance_details.php
error_reporting(E_ALL);
ini_set('display_errors', 1);

// Include required files
require_once 'config.php';
require_once 'contabo_api.php'; // Explicitly include the API class

redirectIfNotLoggedIn();

$message = '';
$error = '';
$instance_details = [];
$snapshots = [];
$available_images = [];
$popular_images = [];

try {
    if (!isset($_GET['instance_id'])) {
        throw new Exception("Instance ID not provided");
    }
    
    $instanceId = $_GET['instance_id'];
    
    // Check if user has access to this instance
    $db = getDB();
    if (!isAdmin()) {
        $checkStmt = $db->prepare("
            SELECT 1 FROM user_vps_assignments 
            WHERE user_id = ? AND instance_id = ?
        ");
        $checkStmt->execute([$_SESSION['user_id'], $instanceId]);
        if (!$checkStmt->fetch()) {
            throw new Exception("You don't have access to this VPS instance");
        }
    }
    
    $contabo = new ContaboAPI();
    
    // Handle actions
    if ($_SERVER['REQUEST_METHOD'] === 'POST') {
        $action = $_POST['action'] ?? '';
        
        switch($action) {
            case 'reinstall':
                $imageId = $_POST['image_id'] ?? '';
                if (empty($imageId)) {
                    throw new Exception("Please select an image to reinstall");
                }
                $contabo->reinstallInstance($instanceId, $imageId);
                $contabo->logAction($instanceId, 'reinstall', "Reinstalled with image: $imageId");
                $message = "Instance reinstallation started successfully!";
                break;
                
            case 'create_snapshot':
                $name = $_POST['snapshot_name'] ?? '';
                $description = $_POST['snapshot_description'] ?? '';
                if (empty($name)) {
                    throw new Exception("Snapshot name is required");
                }
                $result = $contabo->createSnapshot($instanceId, $name, $description);
                $contabo->logAction($instanceId, 'create_snapshot', "Created snapshot: $name");
                $message = "Snapshot created successfully!";
                break;
                
            case 'reset_password':
                $contabo->resetPassword($instanceId);
                $contabo->logAction($instanceId, 'reset_password');
                $message = "Password reset initiated! Check your email for new credentials.";
                break;
                
            case 'rescue':
                $contabo->rescueInstance($instanceId);
                $contabo->logAction($instanceId, 'rescue');
                $message = "Instance booted in rescue mode!";
                break;
                
            case 'cancel':
                $cancelDate = $_POST['cancel_date'] ?? date('Y-m-d', strtotime('+30 days'));
                $contabo->cancelInstance($instanceId, $cancelDate);
                $contabo->logAction($instanceId, 'cancel', "Cancelled with date: $cancelDate");
                $message = "Instance cancellation scheduled successfully!";
                break;
                
            case 'remove_cancellation':
                $contabo->removeCancellation($instanceId);
                $contabo->logAction($instanceId, 'remove_cancellation');
                $message = "Cancellation removal requested! Please contact support if needed.";
                break;
                
            default:
                throw new Exception("Unknown action: $action");
        }
        
        // Redirect to avoid form resubmission
        header("Location: instance_details.php?instance_id=" . $instanceId . "&message=" . urlencode($message));
        exit;
    }
    
    // Handle GET actions
    if (isset($_GET['action'])) {
        $action = $_GET['action'];
        
        switch($action) {
            case 'start':
                $contabo->startInstance($instanceId);
                $contabo->logAction($instanceId, 'start');
                $message = "Instance started successfully!";
                break;
            case 'stop':
                $contabo->stopInstance($instanceId);
                $contabo->logAction($instanceId, 'stop');
                $message = "Instance stopped successfully!";
                break;
            case 'restart':
                $contabo->restartInstance($instanceId);
                $contabo->logAction($instanceId, 'restart');
                $message = "Instance restarted successfully!";
                break;
            case 'shutdown':
                $contabo->shutdownInstance($instanceId);
                $contabo->logAction($instanceId, 'shutdown');
                $message = "Instance shutdown initiated!";
                break;
            case 'delete_snapshot':
                $snapshotId = $_GET['snapshot_id'] ?? '';
                if (!empty($snapshotId)) {
                    $contabo->deleteSnapshot($instanceId, $snapshotId);
                    $contabo->logAction($instanceId, 'delete_snapshot', "Deleted snapshot: $snapshotId");
                    $message = "Snapshot deleted successfully!";
                }
                break;
            case 'rollback_snapshot':
                $snapshotId = $_GET['snapshot_id'] ?? '';
                if (!empty($snapshotId)) {
                    $contabo->rollbackSnapshot($instanceId, $snapshotId);
                    $contabo->logAction($instanceId, 'rollback_snapshot', "Rolled back to snapshot: $snapshotId");
                    $message = "Snapshot rollback initiated successfully!";
                }
                break;
            default:
                throw new Exception("Unknown action: $action");
        }
        
        // Redirect to avoid form resubmission
        header("Location: instance_details.php?instance_id=" . $instanceId . "&message=" . urlencode($message));
        exit;
    }
    
    // Get instance details
    $instance_data = $contabo->getInstanceDetails($instanceId);
    
    if (isset($instance_data['data']) && is_array($instance_data['data']) && count($instance_data['data']) > 0) {
        $instance_details = $instance_data['data'][0];
    } else {
        throw new Exception("Instance not found or no data returned");
    }
    
    // Get snapshots
    try {
        $snapshots_data = $contabo->getSnapshots($instanceId);
        $snapshots = $snapshots_data['data'] ?? [];
    } catch (Exception $e) {
        // If snapshots fail, just log and continue
        error_log("Failed to get snapshots: " . $e->getMessage());
        $snapshots = [];
    }
    
    // Get available images for reinstall - both all and popular
    try {
        $images_data = $contabo->getAvailableImages();
        $available_images = $images_data['data'] ?? [];
        $popular_images = $contabo->getPopularOSImages();
    } catch (Exception $e) {
        // If images fail, just log and continue
        error_log("Failed to get images: " . $e->getMessage());
        $available_images = [];
        $popular_images = [];
    }
    
} catch(Exception $e) {
    $error = $e->getMessage();
    error_log("Error in instance_details.php: " . $e->getMessage());
}

// Check if we have a message from redirect
if (isset($_GET['message'])) {
    $message = htmlspecialchars($_GET['message']);
}

// Format IP addresses properly
$formatted_ips = [];
if (!empty($instance_details)) {
    $contabo = new ContaboAPI();
    $formatted_ips = $contabo->formatIPAddresses($instance_details['ipConfig'] ?? []);
}
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Instance Details - <?php echo htmlspecialchars($instance_details['name'] ?? 'VPS Manager'); ?></title>
    <style>
        body { font-family: Arial, sans-serif; margin: 20px; }
        .message { padding: 10px; margin: 10px 0; border-radius: 4px; }
        .success { background: #d4edda; color: #155724; border: 1px solid #c3e6cb; }
        .error { background: #f8d7da; color: #721c24; border: 1px solid #f5c6cb; }
        .details-grid { display: grid; grid-template-columns: 200px 1fr; gap: 10px; margin: 20px 0; }
        .detail-label { font-weight: bold; }
        .detail-value { padding: 8px; background: #f8f9fa; border-radius: 4px; }
        .btn { display: inline-block; padding: 8px 15px; margin: 5px; border: none; border-radius: 4px; cursor: pointer; text-decoration: none; font-size: 14px; }
        .btn-success { background: #28a745; color: white; }
        .btn-warning { background: #ffc107; color: black; }
        .btn-danger { background: #dc3545; color: white; }
        .btn-info { background: #17a2b8; color: white; }
        .btn-primary { background: #007bff; color: white; }
        .btn-secondary { background: #6c757d; color: white; }
        .section { margin: 30px 0; padding: 20px; border: 1px solid #dee2e6; border-radius: 5px; }
        .section h3 { margin-top: 0; color: #495057; }
        .action-buttons { display: flex; flex-wrap: wrap; gap: 10px; margin: 15px 0; }
        .form-group { margin-bottom: 15px; }
        label { display: block; margin-bottom: 5px; font-weight: bold; }
        input[type="text"], input[type="password"], input[type="date"], select, textarea { 
            width: 100%; max-width: 400px; padding: 8px; border: 1px solid #ddd; border-radius: 4px; 
            display: block;
        }
        .snapshots-grid { display: grid; grid-template-columns: repeat(auto-fill, minmax(300px, 1fr)); gap: 15px; }
        .snapshot-card { padding: 15px; border: 1px solid #dee2e6; border-radius: 5px; background: #f8f9fa; }
        .snapshot-actions { margin-top: 10px; display: flex; gap: 5px; flex-wrap: wrap; }
        .status-running { color: #28a745; font-weight: bold; }
        .status-stopped { color: #dc3545; font-weight: bold; }
        .status-provisioning { color: #ffc107; font-weight: bold; }
        .cancel-warning { background: #fff3cd; border: 1px solid #ffeaa7; color: #856404; padding: 15px; border-radius: 5px; margin: 10px 0; }
        .os-category { margin-bottom: 20px; }
        .os-category h4 { margin-bottom: 10px; color: #495057; border-bottom: 1px solid #dee2e6; padding-bottom: 5px; }
    </style>
</head>
<body>
    <?php if (isAdmin()): ?>
        <a href="view_vps.php" class="btn btn-primary">← Back to VPS List</a>
    <?php else: ?>
        <a href="user_vps.php" class="btn btn-primary">← Back to My VPS</a>
    <?php endif; ?>
    
    <h1>Instance Details: <?php echo htmlspecialchars($instance_details['name'] ?? 'N/A'); ?></h1>
    
    <?php if ($message): ?>
        <div class="message success"><?php echo htmlspecialchars($message); ?></div>
    <?php endif; ?>
    
    <?php if ($error): ?>
        <div class="message error"><?php echo htmlspecialchars($error); ?></div>
    <?php endif; ?>
    
    <?php if (!empty($instance_details)): ?>
        <!-- Instance Information Section -->
        <div class="section">
            <h3>Instance Information</h3>
            <div class="details-grid">
                <div class="detail-label">Instance ID:</div>
                <div class="detail-value"><?php echo htmlspecialchars($instance_details['instanceId'] ?? 'N/A'); ?></div>
                
                <div class="detail-label">Name:</div>
                <div class="detail-value"><?php echo htmlspecialchars($instance_details['name'] ?? 'N/A'); ?></div>
                
                <div class="detail-label">Display Name:</div>
                <div class="detail-value"><?php echo htmlspecialchars($instance_details['displayName'] ?? 'N/A'); ?></div>
                
                <div class="detail-label">Status:</div>
                <div class="detail-value">
                    <span class="status-<?php echo strtolower($instance_details['status'] ?? ''); ?>">
                        <?php echo htmlspecialchars($instance_details['status'] ?? 'Unknown'); ?>
                    </span>
                </div>
                
                <div class="detail-label">Region:</div>
                <div class="detail-value"><?php echo htmlspecialchars($instance_details['region'] ?? 'N/A'); ?></div>
                
                <div class="detail-label">Product Type:</div>
                <div class="detail-value"><?php echo htmlspecialchars($instance_details['productType'] ?? 'N/A'); ?></div>
                
                <div class="detail-label">Created Date:</div>
                <div class="detail-value">
                    <?php 
                    if (isset($instance_details['createdDate'])) {
                        echo date('M j, Y H:i:s', strtotime($instance_details['createdDate']));
                    } else {
                        echo 'N/A';
                    }
                    ?>
                </div>
                
                <!-- Fixed IP Address Display -->
                <div class="detail-label">IPv4 Addresses:</div>
                <div class="detail-value">
                    <?php if (!empty($formatted_ips['ipv4'])): ?>
                        <?php foreach ($formatted_ips['ipv4'] as $ip): ?>
                            <div><?php echo htmlspecialchars($ip); ?></div>
                        <?php endforeach; ?>
                    <?php else: ?>
                        No IPv4 addresses found
                    <?php endif; ?>
                </div>
                
                <div class="detail-label">IPv6 Addresses:</div>
                <div class="detail-value">
                    <?php if (!empty($formatted_ips['ipv6'])): ?>
                        <?php foreach ($formatted_ips['ipv6'] as $ip): ?>
                            <div><?php echo htmlspecialchars($ip); ?></div>
                        <?php endforeach; ?>
                    <?php else: ?>
                        No IPv6 addresses found
                    <?php endif; ?>
                </div>
                
                <?php if (isset($instance_details['imageId'])): ?>
                    <div class="detail-label">Image ID:</div>
                    <div class="detail-value"><?php echo htmlspecialchars($instance_details['imageId']); ?></div>
                <?php endif; ?>
                
                <?php if (isset($instance_details['productId'])): ?>
                    <div class="detail-label">Product ID:</div>
                    <div class="detail-value"><?php echo htmlspecialchars($instance_details['productId']); ?></div>
                <?php endif; ?>
            </div>
        </div>

        <!-- Power Management Section -->
        <div class="section">
            <h3>Power Management</h3>
            <div class="action-buttons">
                <a href="instance_details.php?instance_id=<?php echo $instanceId; ?>&action=start" class="btn btn-success" onclick="return confirm('Start this instance?')">Start</a>
                <a href="instance_details.php?instance_id=<?php echo $instanceId; ?>&action=stop" class="btn btn-danger" onclick="return confirm('Stop this instance?')">Stop</a>
                <a href="instance_details.php?instance_id=<?php echo $instanceId; ?>&action=restart" class="btn btn-warning" onclick="return confirm('Restart this instance?')">Restart</a>
                <a href="instance_details.php?instance_id=<?php echo $instanceId; ?>&action=shutdown" class="btn btn-secondary" onclick="return confirm('Shutdown this instance?')">Shutdown</a>
            </div>
        </div>

        <!-- Reinstall OS Section -->
        <?php if (!empty($available_images)): ?>
        <div class="section">
            <h3>Reinstall Operating System</h3>
            
            <!-- Popular OS Images -->
            <?php if (!empty($popular_images)): ?>
            <div class="os-category">
                <h4>Popular Operating Systems</h4>
                <form method="POST">
                    <input type="hidden" name="action" value="reinstall">
                    <div class="form-group">
                        <select name="image_id" required style="width: 100%; max-width: 400px; padding: 8px;">
                            <option value="">-- Select Popular OS --</option>
                            <?php foreach ($popular_images as $image): ?>
                                <?php if (isset($image['imageId']) && isset($image['name'])): ?>
                                    <option value="<?php echo htmlspecialchars($image['imageId']); ?>">
                                        <?php echo htmlspecialchars($image['name']); ?> 
                                        (<?php echo htmlspecialchars($image['osType'] ?? 'Unknown'); ?>)
                                    </option>
                                <?php endif; ?>
                            <?php endforeach; ?>
                        </select>
                    </div>
                    <button type="submit" class="btn btn-warning" onclick="return confirm('WARNING: This will erase all data on the instance. Continue?')">
                        Reinstall OS
                    </button>
                </form>
            </div>
            <?php endif; ?>
            
            <!-- All Available Images -->
            <div class="os-category">
                <h4>All Available Images</h4>
                <form method="POST">
                    <input type="hidden" name="action" value="reinstall">
                    <div class="form-group">
                        <select name="image_id" required style="width: 100%; max-width: 400px; padding: 8px;">
                            <option value="">-- Select Any Available Image --</option>
                            <?php foreach ($available_images as $image): ?>
                                <?php if (isset($image['imageId']) && isset($image['name'])): ?>
                                    <option value="<?php echo htmlspecialchars($image['imageId']); ?>">
                                        <?php echo htmlspecialchars($image['name']); ?> 
                                        (<?php echo htmlspecialchars($image['osType'] ?? 'Unknown'); ?>)
                                    </option>
                                <?php endif; ?>
                            <?php endforeach; ?>
                        </select>
                    </div>
                    <button type="submit" class="btn btn-warning" onclick="return confirm('WARNING: This will erase all data on the instance. Continue?')">
                        Reinstall OS
                    </button>
                </form>
            </div>
        </div>
        <?php endif; ?>

        <!-- Password & Rescue Section -->
        <div class="section">
            <h3>Password & Rescue Mode</h3>
            <div class="action-buttons">
                <form method="POST" style="display: inline;">
                    <input type="hidden" name="action" value="reset_password">
                    <button type="submit" class="btn btn-info" onclick="return confirm('Reset password for this instance?')">
                        Reset Password
                    </button>
                </form>
                
                <form method="POST" style="display: inline;">
                    <input type="hidden" name="action" value="rescue">
                    <button type="submit" class="btn btn-secondary" onclick="return confirm('Boot this instance in rescue mode?')">
                        Boot in Rescue Mode
                    </button>
                </form>
            </div>
        </div>

        <!-- Cancel Instance Section -->
        <div class="section">
            <h3>Cancel Instance</h3>
            <div class="cancel-warning">
                <strong>Warning:</strong> Cancelling this instance will permanently delete it and all associated data. 
                This action cannot be undone. Make sure you have backups of any important data.
            </div>
            <form method="POST">
                <input type="hidden" name="action" value="cancel">
                <div class="form-group">
                    <label for="cancel_date">Cancellation Date:</label>
                    <input type="date" id="cancel_date" name="cancel_date" 
                           value="<?php echo date('Y-m-d', strtotime('+30 days')); ?>"
                           min="<?php echo date('Y-m-d'); ?>" 
                           style="width: 100%; max-width: 400px;">
                    <small>Select the date when the instance should be cancelled</small>
                </div>
                <button type="submit" class="btn btn-danger" onclick="return confirm('WARNING: This will permanently cancel and delete this instance. Are you absolutely sure?')">
                    Cancel Instance
                </button>
            </form>
            
            <!-- Remove Cancellation Option -->
            <div style="margin-top: 20px; padding-top: 20px; border-top: 1px solid #dee2e6;">
                <h4>Remove Cancellation</h4>
                <form method="POST">
                    <input type="hidden" name="action" value="remove_cancellation">
                    <button type="submit" class="btn btn-info" onclick="return confirm('Remove cancellation for this instance? Note: This may require contacting support.')">
                        Remove Cancellation
                    </button>
                </form>
            </div>
        </div>

        <!-- Snapshots Section -->
        <div class="section">
            <h3>Snapshots</h3>
            
            <!-- Create Snapshot Form -->
            <div style="margin-bottom: 20px; padding: 15px; background: #e9ecef; border-radius: 5px;">
                <h4>Create New Snapshot</h4>
                <form method="POST">
                    <input type="hidden" name="action" value="create_snapshot">
                    <div class="form-group">
                        <label for="snapshot_name">Snapshot Name:</label>
                        <input type="text" id="snapshot_name" name="snapshot_name" required maxlength="30" style="width: 100%; max-width: 400px;">
                    </div>
                    <div class="form-group">
                        <label for="snapshot_description">Description:</label>
                        <textarea id="snapshot_description" name="snapshot_description" rows="3" maxlength="255" style="width: 100%; max-width: 400px;"></textarea>
                    </div>
                    <button type="submit" class="btn btn-primary">Create Snapshot</button>
                </form>
            </div>

            <!-- Existing Snapshots -->
            <h4>Existing Snapshots</h4>
            <?php if (empty($snapshots)): ?>
                <p>No snapshots found for this instance.</p>
            <?php else: ?>
                <div class="snapshots-grid">
                    <?php foreach ($snapshots as $snapshot): ?>
                        <?php if (isset($snapshot['snapshotId']) && isset($snapshot['name'])): ?>
                            <div class="snapshot-card">
                                <strong><?php echo htmlspecialchars($snapshot['name']); ?></strong>
                                <?php if (!empty($snapshot['description'])): ?>
                                    <p><?php echo htmlspecialchars($snapshot['description']); ?></p>
                                <?php endif; ?>
                                <small>
                                    Created: 
                                    <?php 
                                    if (isset($snapshot['createdDate'])) {
                                        echo date('M j, Y H:i', strtotime($snapshot['createdDate']));
                                    } else {
                                        echo 'Unknown';
                                    }
                                    ?>
                                </small>
                                <div class="snapshot-actions">
                                    <a href="instance_details.php?instance_id=<?php echo $instanceId; ?>&action=rollback_snapshot&snapshot_id=<?php echo $snapshot['snapshotId']; ?>" 
                                       class="btn btn-warning btn-sm" 
                                       onclick="return confirm('Rollback to this snapshot? This will restore the instance to this state.')">
                                        Rollback
                                    </a>
                                    <a href="instance_details.php?instance_id=<?php echo $instanceId; ?>&action=delete_snapshot&snapshot_id=<?php echo $snapshot['snapshotId']; ?>" 
                                       class="btn btn-danger btn-sm" 
                                       onclick="return confirm('Delete this snapshot?')">
                                        Delete
                                    </a>
                                </div>
                            </div>
                        <?php endif; ?>
                    <?php endforeach; ?>
                </div>
            <?php endif; ?>
        </div>

    <?php else: ?>
        <div class="message error">No instance details found.</div>
    <?php endif; ?>
</body>
</html>