<?php
// customer_vps.php
require_once 'config.php';
redirectIfNotLoggedIn();
redirectIfNotCustomer();

$message = '';
$error = '';
$customer_instances = [];

try {
    $db = getDB();
    
    // Get VPS instances assigned to the current customer
    $stmt = $db->prepare("
        SELECT vi.* 
        FROM vps_instances vi 
        INNER JOIN user_vps_assignments uva ON vi.instance_id = uva.instance_id 
        WHERE uva.customer_id = ? 
        ORDER BY vi.created_date DESC
    ");
    $stmt->execute([$_SESSION['user_id']]);
    $customer_instances = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
    // Handle instance actions for customers
    if (isset($_GET['action']) && isset($_GET['instance_id'])) {
        $action = $_GET['action'];
        $instanceId = $_GET['instance_id'];
        
        // Verify customer has access to this instance
        $checkStmt = $db->prepare("
            SELECT 1 FROM user_vps_assignments 
            WHERE customer_id = ? AND instance_id = ?
        ");
        $checkStmt->execute([$_SESSION['user_id'], $instanceId]);
        
        if ($checkStmt->fetch()) {
            $contabo = new ContaboAPI();
            
            switch($action) {
                case 'start':
                    $contabo->startInstance($instanceId);
                    $contabo->logAction($instanceId, 'start');
                    $message = "Instance started successfully!";
                    break;
                case 'stop':
                    $contabo->stopInstance($instanceId);
                    $contabo->logAction($instanceId, 'stop');
                    $message = "Instance stopped successfully!";
                    break;
                case 'restart':
                    $contabo->restartInstance($instanceId);
                    $contabo->logAction($instanceId, 'restart');
                    $message = "Instance restarted successfully!";
                    break;
            }
        } else {
            $error = "You don't have permission to perform this action on this instance.";
        }
    }
    
} catch(Exception $e) {
    $error = $e->getMessage();
}
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>My VPS - <?php echo htmlspecialchars($_SESSION['username']); ?></title>
    <style>
        body { font-family: Arial, sans-serif; margin: 20px; }
        .message { padding: 10px; margin: 10px 0; border-radius: 4px; }
        .success { background: #d4edda; color: #155724; border: 1px solid #c3e6cb; }
        .error { background: #f8d7da; color: #721c24; border: 1px solid #f5c6cb; }
        table { width: 100%; border-collapse: collapse; margin-top: 20px; }
        th, td { padding: 12px; text-align: left; border-bottom: 1px solid #ddd; }
        th { background-color: #f8f9fa; font-weight: bold; }
        .status-running { color: #28a745; font-weight: bold; }
        .status-stopped { color: #dc3545; font-weight: bold; }
        .status-provisioning { color: #ffc107; font-weight: bold; }
        .btn { display: inline-block; padding: 6px 12px; margin: 2px; border: none; border-radius: 4px; cursor: pointer; text-decoration: none; font-size: 12px; }
        .btn-success { background: #28a745; color: white; }
        .btn-warning { background: #ffc107; color: black; }
        .btn-danger { background: #dc3545; color: white; }
        .btn-info { background: #17a2b8; color: white; }
        .header { display: flex; justify-content: space-between; align-items: center; margin-bottom: 20px; flex-wrap: wrap; }
        .actions { display: flex; gap: 5px; flex-wrap: wrap; }
        .no-vps { text-align: center; padding: 40px; color: #6c757d; }
    </style>
</head>
<body>
    <div class="header">
        <h1>My VPS Instances</h1>
        <div>
            <span>Welcome, <?php echo htmlspecialchars($_SESSION['username']); ?> (Customer)</span>
            <a href="logout.php" class="btn btn-secondary">Logout</a>
        </div>
    </div>
    
    <?php if ($message): ?>
        <div class="message success"><?php echo htmlspecialchars($message); ?></div>
    <?php endif; ?>
    
    <?php if ($error): ?>
        <div class="message error"><?php echo htmlspecialchars($error); ?></div>
    <?php endif; ?>
    
    <?php if (empty($customer_instances)): ?>
        <div class="no-vps">
            <h3>No VPS Assigned</h3>
            <p>You don't have any VPS instances assigned to your account yet.</p>
            <p>Please contact your store owner to assign VPS instances to your account.</p>
        </div>
    <?php else: ?>
        <table>
            <thead>
                <tr>
                    <th>Name</th>
                    <th>Instance ID</th>
                    <th>Status</th>
                    <th>IP Address</th>
                    <th>Product Type</th>
                    <th>Assigned Date</th>
                    <th>Actions</th>
                </tr>
            </thead>
            <tbody>
                <?php foreach ($customer_instances as $instance): ?>
                <tr>
                    <td><?php echo htmlspecialchars($instance['name']); ?></td>
                    <td><?php echo htmlspecialchars($instance['instance_id']); ?></td>
                    <td class="status-<?php echo strtolower($instance['status']); ?>">
                        <?php echo htmlspecialchars($instance['status']); ?>
                    </td>
                    <td><?php echo htmlspecialchars($instance['ip_address']); ?></td>
                    <td><?php echo htmlspecialchars($instance['product_type']); ?></td>
                    <td><?php echo date('M j, Y H:i', strtotime($instance['created_date'])); ?></td>
                    <td class="actions">
                        <a href="customer_vps.php?action=start&instance_id=<?php echo $instance['instance_id']; ?>" class="btn btn-success" onclick="return confirm('Start this instance?')">Start</a>
                        <a href="customer_vps.php?action=stop&instance_id=<?php echo $instance['instance_id']; ?>" class="btn btn-danger" onclick="return confirm('Stop this instance?')">Stop</a>
                        <a href="customer_vps.php?action=restart&instance_id=<?php echo $instance['instance_id']; ?>" class="btn btn-warning" onclick="return confirm('Restart this instance?')">Restart</a>
                        <a href="instance_details.php?instance_id=<?php echo $instance['instance_id']; ?>" class="btn btn-info">View Details</a>
                    </td>
                </tr>
                <?php endforeach; ?>
            </tbody>
        </table>
    <?php endif; ?>
</body>
</html>